// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package f3

import (
	"context"
	"path/filepath"
	"testing"

	filesystem_options "code.forgejo.org/f3/gof3/v3/forges/filesystem/options"
	"code.forgejo.org/f3/gof3/v3/id"
	"code.forgejo.org/f3/gof3/v3/options"
	"code.forgejo.org/f3/gof3/v3/path"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	tests_forge "code.forgejo.org/f3/gof3/v3/tree/tests/f3/forge"

	"github.com/stretchr/testify/assert"
)

func TestF3FilesystemMappedID(t *testing.T) {
	ctx := context.Background()

	//
	// aTree only has /forge/user/10111
	//
	aTree := generic.GetFactory("f3")(ctx, tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t))
	aDir := aTree.GetOptions().(options.URLInterface).GetURL()

	creator := NewCreator(t, "F3", aTree.GetLogger())

	aF3Tree := aTree.(f3_tree.TreeInterface)

	userID := "10111"

	aF3Tree.CreateChild(ctx, "/", func(parent path.Path, forge generic.NodeInterface) {
		forge.FromFormat(creator.GenerateForge())
	})

	aF3Tree.CreateChild(ctx, "/forge/users", func(parent path.Path, user generic.NodeInterface) {
		user.FromFormat(GeneratorSetID(creator.GenerateUser(), userID))
	})

	//
	// bTree mirrors aTree exactly
	//
	rootPath := generic.NewPathFromString("")

	bTree := generic.GetFactory("f3")(ctx, tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t))

	generic.TreeMirror(ctx, aTree, bTree, rootPath, generic.NewMirrorOptions())

	assert.True(t, generic.TreeCompare(ctx, aTree, rootPath, bTree, rootPath))

	//
	// aTree maps user id 10111 to 10111.mapped
	//
	userPath := generic.NewPathFromString(filepath.Join("/forge/users", userID))

	userMappedID := id.NewNodeID("10111.mapped")

	assert.True(t, aTree.Apply(ctx, userPath, generic.NewApplyOptions(func(ctx context.Context, parent, path path.Path, node generic.NodeInterface) {
		node.SetMappedID(userMappedID)
		node.Upsert(ctx)
	})))

	aTree = generic.GetFactory("f3")(ctx, tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t))
	aTree.GetOptions().(options.URLInterface).SetURL(aDir)

	aTree.WalkAndGet(ctx, generic.NewWalkOptions(nil))
	assert.NotEqualValues(t, generic.NilNode, aTree.Find(userPath))

	//
	// cTree mirrors aTree with user id 10111 remapped to 10111.mapped
	//
	cTree := generic.GetFactory("f3")(ctx, tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t))
	cDir := cTree.GetOptions().(options.URLInterface).GetURL()

	generic.TreeMirror(ctx, aTree, cTree, rootPath, generic.NewMirrorOptions())

	userMappedPath := generic.NewPathFromString(filepath.Join("/forge/users", userMappedID.String()))

	assert.NotEqualValues(t, generic.NilNode, cTree.Find(userMappedPath))
	assert.EqualValues(t, generic.NilNode, cTree.Find(userPath))

	//
	// reset cTree and read from the filesystem
	//
	cTree = generic.GetFactory("f3")(ctx, tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t))
	cTree.GetOptions().(options.URLInterface).SetURL(cDir)

	cTree.WalkAndGet(ctx, generic.NewWalkOptions(nil))

	assert.NotEqualValues(t, generic.NilNode, cTree.Find(userMappedPath))
	assert.EqualValues(t, generic.NilNode, cTree.Find(userPath))

	//
	// delete aTree user
	//
	deleted := aTree.Find(userPath).Delete(ctx)
	assert.EqualValues(t, userMappedID, deleted.GetMappedID())
	assert.EqualValues(t, generic.NilNode, cTree.Find(userPath))
}
